"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StatRead = void 0;
const bin_util_1 = require("@ot-builder/bin-util");
const ImpLib = require("@ot-builder/common-impl");
const errors_1 = require("@ot-builder/errors");
const ot_name_1 = require("@ot-builder/ot-name");
const primitive_1 = require("@ot-builder/primitive");
exports.StatRead = (0, bin_util_1.Read)(view => {
    const majorVersion = view.uint16();
    const minorVersion = view.uint16();
    errors_1.Assert.SubVersionSupported("STATTable", majorVersion, minorVersion, [1, 1], [1, 2]);
    const designAxisSize = view.uint16();
    errors_1.Assert.SizeMatch("STATTable::designAxisSize", designAxisSize, primitive_1.Tag.size + primitive_1.UInt16.size * 2);
    const designAxisCount = view.uint16();
    const vwAxisRecord = view.ptr32Nullable();
    const axisValueCount = view.uint16();
    const vwAxisValues = view.ptr32Nullable();
    const elidedFallbackNameID = view.uint16();
    const axes = vwAxisRecord ? vwAxisRecord.next(DesignAxisArray, designAxisCount) : [];
    const values = vwAxisValues
        ? vwAxisValues.next(AxisValueArray, axisValueCount, ImpLib.Order.fromList(`Axes`, axes))
        : [];
    const table = new ot_name_1.Stat.Table(axes, values, elidedFallbackNameID);
    return table;
});
const DesignAxisArray = (0, bin_util_1.Read)((view, count) => {
    const arr = [];
    for (let aid = 0; aid < count; aid++) {
        const tag = view.next(primitive_1.Tag);
        const axisNameID = view.uint16();
        const axisOrdering = view.uint16();
        arr.push(new ot_name_1.Stat.Axis(tag, axisNameID, axisOrdering));
    }
    return arr;
});
const AxisValueArray = (0, bin_util_1.Read)((view, count, axes) => {
    const arr = [];
    for (let aid = 0; aid < count; aid++) {
        arr.push(view.ptr16().next(AxisValue, axes));
    }
    return arr;
});
const AxisValue = (0, bin_util_1.Read)((view, axes) => {
    const format = view.lift(0).uint16();
    switch (format) {
        case 1:
            return view.next(AxisValueFormat1, axes);
        case 2:
            return view.next(AxisValueFormat2, axes);
        case 3:
            return view.next(AxisValueFormat3, axes);
        case 4:
            return view.next(AxisValueFormat4, axes);
        default:
            throw errors_1.Errors.FormatNotSupported(`STATTable::AxisValue`, format);
    }
});
const AxisValueFormat1 = (0, bin_util_1.Read)((view, axes) => {
    const format = view.uint16();
    errors_1.Assert.FormatSupported(`STATTable::AxisValueFormat1`, format, 1);
    const axisIndex = view.uint16();
    const axis = axes.at(axisIndex);
    const flags = view.uint16();
    const valueNameID = view.uint16();
    const value = view.next(primitive_1.F16D16);
    return ImpLib.Tuple.Tie(new ot_name_1.Stat.AxisValue.Static(axis, value), new ot_name_1.Stat.NameAssignment(flags, valueNameID));
});
const AxisValueFormat2 = (0, bin_util_1.Read)((view, axes) => {
    const format = view.uint16();
    errors_1.Assert.FormatSupported(`STATTable::AxisValueFormat2`, format, 2);
    const axisIndex = view.uint16();
    const axis = axes.at(axisIndex);
    const flags = view.uint16();
    const valueNameID = view.uint16();
    const nominalValue = view.next(primitive_1.F16D16);
    const rangeMinValue = view.next(primitive_1.F16D16);
    const rangeMaxValue = view.next(primitive_1.F16D16);
    return ImpLib.Tuple.Tie(new ot_name_1.Stat.AxisValue.Variable(axis, rangeMinValue, nominalValue, rangeMaxValue), new ot_name_1.Stat.NameAssignment(flags, valueNameID));
});
const AxisValueFormat3 = (0, bin_util_1.Read)((view, axes) => {
    const format = view.uint16();
    errors_1.Assert.FormatSupported(`STATTable::AxisValueFormat3`, format, 3);
    const axisIndex = view.uint16();
    const axis = axes.at(axisIndex);
    const flags = view.uint16();
    const valueNameID = view.uint16();
    const value = view.next(primitive_1.F16D16);
    const linkedValue = view.next(primitive_1.F16D16);
    return ImpLib.Tuple.Tie(new ot_name_1.Stat.AxisValue.Linked(axis, value, linkedValue), new ot_name_1.Stat.NameAssignment(flags, valueNameID));
});
const AxisValueFormat4 = (0, bin_util_1.Read)((view, axes) => {
    const format = view.uint16();
    errors_1.Assert.FormatSupported(`STATTable::AxisValueFormat4`, format, 4);
    const axisCount = view.uint16();
    const flags = view.uint16();
    const valueNameID = view.uint16();
    const assignments = [];
    for (let aid = 0; aid < axisCount; aid++) {
        const axisIndex = view.uint16();
        const axis = axes.at(axisIndex);
        const value = view.next(primitive_1.F16D16);
        assignments.push([axis, value]);
    }
    return ImpLib.Tuple.Tie(new ot_name_1.Stat.AxisValue.PolyAxis(assignments), new ot_name_1.Stat.NameAssignment(flags, valueNameID));
});
//# sourceMappingURL=read.js.map