/*
 *  $Id: xyz_merge.c 28808 2025-11-05 18:26:20Z yeti-dn $
 *  Copyright (C) 2019-2025 David Necas (Yeti).
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "config.h"
#include <glib/gi18n-lib.h>
#include <string.h>
#include <stdlib.h>
#include <gtk/gtk.h>
#include <gwy.h>

#define RUN_MODES (GWY_RUN_INTERACTIVE | GWY_RUN_IMMEDIATE)

enum {
    PARAM_DO_AVERAGE,
    PARAM_OTHER_SURFACE,
};

typedef struct {
    GwyParams *params;
    GwySurface *surface;
    GwySurface *result;
} ModuleArgs;

typedef struct {
    ModuleArgs *args;
    GtkWidget *dialog;
    GwyParamTable *table;
} ModuleGUI;

static gboolean         module_register     (void);
static GwyParamDef*     define_module_params(void);
static void             module_main         (GwyFile *data,
                                             GwyRunModeFlags mode);
static GwyDialogOutcome run_gui             (ModuleArgs *arg);
static gboolean         merge_data_filter   (GwyFile *data,
                                             gint id,
                                             gpointer user_data);
static gboolean         execute             (ModuleArgs *args,
                                             GwyFile *data,
                                             gint id);

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Elementary XYZ data operations."),
    "Yeti <yeti@gwyddion.net>",
    "2.0",
    "David Nečas (Yeti)",
    "2018",
};

GWY_MODULE_QUERY2(module_info, xyz_merge)

static gboolean
module_register(void)
{
    gwy_xyz_func_register("xyz_merge",
                          module_main,
                          N_("/_Merge..."),
                          NULL,
                          RUN_MODES,
                          GWY_MENU_FLAG_XYZ,
                          N_("Merge two XYZ point sets"));

    return TRUE;
}

static GwyParamDef*
define_module_params(void)
{
    static GwyParamDef *paramdef = NULL;

    if (paramdef)
        return paramdef;

    paramdef = gwy_param_def_new();
    gwy_param_def_set_function_name(paramdef, gwy_xyz_func_current());
    gwy_param_def_add_xyz_id(paramdef, PARAM_OTHER_SURFACE, "op2", _("Second _XYZ data"));
    gwy_param_def_add_boolean(paramdef, PARAM_DO_AVERAGE, "do_average", _("_Average coincident points"), TRUE);
    return paramdef;
}

static void
module_main(GwyFile *data, GwyRunModeFlags mode)
{
    g_return_if_fail(mode & RUN_MODES);

    ModuleArgs args;
    gint id;
    gwy_clear1(args);

    args.params = gwy_params_new_from_settings(define_module_params());

    gwy_data_browser_get_current(GWY_APP_SURFACE, &args.surface,
                                 GWY_APP_SURFACE_ID, &id,
                                 0);
    g_return_if_fail(GWY_IS_SURFACE(args.surface));

    GwyDialogOutcome outcome = GWY_DIALOG_PROCEED;
    if (mode == GWY_RUN_INTERACTIVE) {
        outcome = run_gui(&args);
        gwy_params_save_to_settings(args.params);
        if (outcome == GWY_DIALOG_CANCEL)
            goto end;
    }
    if (outcome != GWY_DIALOG_HAVE_RESULT) {
        if (!execute(&args, data, id))
            goto end;
    }

    gint newid = gwy_file_add_xyz(data, args.result);

    gwy_file_set_visible(data, GWY_FILE_XYZ, newid, TRUE);
    gwy_file_set_title(data, GWY_FILE_XYZ, newid, _("Merged"), TRUE);
    gwy_file_sync_items(data, GWY_FILE_XYZ, id,
                        data, GWY_FILE_XYZ, newid,
                        GWY_FILE_ITEM_PALETTE, FALSE);

    g_object_unref(args.result);

end:
    g_object_unref(args.params);
}

static GwyDialogOutcome
run_gui(ModuleArgs *args)
{
    ModuleGUI gui;
    gwy_clear1(gui);
    gui.args = args;

    gui.dialog = gwy_dialog_new(_("Merge XYZ Data"));
    GwyDialog *dialog = GWY_DIALOG(gui.dialog);
    gwy_dialog_add_buttons(dialog, GTK_RESPONSE_CANCEL, GTK_RESPONSE_OK, 0);

    GwyParamTable *table = gui.table = gwy_param_table_new(args->params);
    gwy_param_table_append_xyz_id(table, PARAM_OTHER_SURFACE);
    gwy_param_table_data_id_set_filter(table, PARAM_OTHER_SURFACE, merge_data_filter, &gui, NULL);
    gwy_param_table_append_checkbox(table, PARAM_DO_AVERAGE);
    gwy_dialog_add_param_table(dialog, table);
    gwy_dialog_add_content(dialog, gwy_param_table_widget(table), TRUE, TRUE, 0);

    return gwy_dialog_run(dialog);
}

static gboolean
merge_data_filter(GwyFile *data, gint id, gpointer user_data)
{
    ModuleGUI *gui = (ModuleGUI*)user_data;
    GwySurface *surface = gui->args->surface;
    GwySurface *othersurface = gwy_file_get_xyz(data, id);

    if (othersurface == surface)
        return FALSE;

    GwyUnit *unit1 = gwy_surface_get_unit_xy(surface);
    GwyUnit *unit2 = gwy_surface_get_unit_xy(othersurface);
    if (!gwy_unit_equal(unit1, unit2))
        return FALSE;

    unit1 = gwy_surface_get_unit_z(surface);
    unit2 = gwy_surface_get_unit_z(othersurface);
    if (!gwy_unit_equal(unit1, unit2))
        return FALSE;

    return TRUE;
}

static int
compare_xy(gconstpointer a, gconstpointer b)
{
    return memcmp(a, b, sizeof(GwyXY));
}

/* Merge exact matches.  We do not promise anything cleverer. */
static guint
merge_coninciding_xyz(GwyXYZ *xyz, guint n)
{
    qsort(xyz, n, sizeof(GwyXYZ), compare_xy);

    guint len, bstart = 0;
    guint i, pos = 0;
    for (i = 1; i < n; i++) {
        if (xyz[i].x != xyz[bstart].x || xyz[i].y != xyz[bstart].y) {
            len = i-bstart;
            xyz[pos] = xyz[bstart++];
            while (bstart < i)
                xyz[pos].z += xyz[bstart++].z;
            xyz[pos++].z /= len;
        }
    }
    len = i - bstart;
    xyz[pos] = xyz[bstart++];
    while (bstart < i)
        xyz[pos].z += xyz[bstart++].z;
    xyz[pos++].z /= len;

    gwy_debug("merged %u points", n-pos);
    return pos;
}

static gboolean
execute(ModuleArgs *args, GwyFile *data, gint id)
{
    GwySurface *surface = args->surface;
    GwySurface *surface2 = gwy_params_get_xyz(args->params, PARAM_OTHER_SURFACE);
    if (!surface2)
        return FALSE;

    gboolean do_average = gwy_params_get_boolean(args->params, PARAM_DO_AVERAGE);
    GwySurface *out = args->result = gwy_surface_new_alike(surface);
    guint n1 = gwy_surface_get_npoints(surface);
    guint n2 = gwy_surface_get_npoints(surface2);
    const GwyXYZ *xyz1 = gwy_surface_get_data_const(surface);
    const GwyXYZ *xyz2 = gwy_surface_get_data_const(surface2);
    GwyXYZ *xyz = g_new(GwyXYZ, n1+n2);

    gwy_app_wait_start(gwy_data_browser_get_window_for_data(data, GWY_FILE_XYZ, id), _("Merging..."));

    gwy_assign(xyz, xyz1, n1);
    gwy_assign(xyz + n1, xyz2, n2);
    guint n = n1 + n2;
    if (do_average) {
        n = merge_coninciding_xyz(xyz, n);
    }
    gwy_app_wait_finish();
    gwy_surface_set_data_full(out, xyz, n);
    g_free(xyz);

    return TRUE;
}

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
