/*
 * Copyright (c) 2013, 2025, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package jdk.jfr.event.runtime;

import static jdk.test.lib.Asserts.assertFalse;
import static jdk.test.lib.Asserts.assertTrue;

import java.time.Duration;
import java.util.concurrent.CountDownLatch;

import jdk.jfr.Recording;
import jdk.jfr.consumer.RecordedEvent;
import jdk.test.lib.jfr.EventNames;
import jdk.test.lib.jfr.Events;
import jdk.test.lib.thread.TestThread;
import jdk.test.lib.thread.XRun;

/**
 * @test
 * @requires vm.flagless
 * @requires vm.hasJFR
 * @library /test/lib
 * @run main/othervm jdk.jfr.event.runtime.TestJavaMonitorWaitEvent
 */
public class TestJavaMonitorWaitEvent {

    private final static String EVENT_NAME = EventNames.JavaMonitorWait;
    private static final long WAIT_TIME = 123456;

    static class Lock {
    }

    static boolean silenceFindBugsNakedNotify;

    public static void main(String[] args) throws Throwable {
        Recording recording = new Recording();
        recording.enable(EVENT_NAME).withThreshold(Duration.ofMillis(0));

        final Lock lock = new Lock();
        final CountDownLatch latch = new CountDownLatch(1);

        TestThread waitThread = new TestThread(new XRun() {
            @Override
            public void xrun() throws Throwable {
                synchronized (lock) {
                    latch.countDown();
                    lock.wait(WAIT_TIME);
                    silenceFindBugsNakedNotify = false;
                }
            }
        });

        try {
            recording.start();
            waitThread.start();
            latch.await();
            synchronized (lock) {
                silenceFindBugsNakedNotify = true;
                lock.notifyAll();
            }
        } finally {
            waitThread.join();
            recording.stop();
        }

        boolean isAnyFound = false;
        for (RecordedEvent event : Events.fromRecording(recording)) {
            System.out.println("Event:" + event);
            if ((long)Events.assertField(event, "timeout").getValue() != WAIT_TIME) {
                continue;
            }

            assertFalse(isAnyFound, "Found more than 1 event");
            isAnyFound = true;
            Events.assertEventThread(event, waitThread);
            final String lockClassName = lock.getClass().getName().replace('.', '/');
            Events.assertField(event, "monitorClass.name").equal(lockClassName);
            Events.assertField(event, "address").notEqual(0L);
            Events.assertField(event, "timedOut").equal(false);
            Events.assertEventThread(event, "notifier", Thread.currentThread());
        }
        assertTrue(isAnyFound, "Correct event not found");
    }
}
