/*
   Copyright 2005-2010 Jakub Kruszona-Zawadzki, Gemius SA
   Copyright 2013-2014 EditShare
   Copyright 2013-2016 Skytechnology sp. z o.o.
   Copyright 2023      Leil Storage OÜ


   SaunaFS is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, version 3.

   SaunaFS is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with SaunaFS. If not, see <http://www.gnu.org/licenses/>.
 */

#include "common/platform.h"

#include <algorithm>
#include <cstdint>
#include <stdio.h>
#include <stdlib.h>

#include "common/datapack.h"
#include "errors/saunafs_error_codes.h"
#include "errors/sfserr.h"
#include "tools/tools_commands.h"
#include "tools/tools_common_functions.h"

static void get_trashtime_usage() {
	fprintf(stderr,
	        "get objects trashtime (how many seconds file should be left in trash)\n\nusage: "
	        "\n saunafs gettrashtime [-nhHr] name [name ...]\n");
	print_numberformat_options();
	print_recursive_option();
}

static int get_trashtime(const char *fname, uint8_t mode) {
	uint32_t cmd, leng;
	inode_t inode;
	uint32_t fn, dn, i;
	uint32_t trashtime;
	uint32_t cnt;
	int fd;
	fd = open_master_conn(fname, &inode, nullptr, false);
	if (fd < 0) {
		return -1;
	}

	constexpr uint32_t kGetTrashTimePayloadSize = sizeof(uint32_t) + sizeof(inode) + sizeof(mode);
	constexpr uint32_t kReqBuffSize =
	    sizeof(cmd) + sizeof(kGetTrashTimePayloadSize) + kGetTrashTimePayloadSize;
	uint8_t reqbuff[kReqBuffSize], *wptr, *buff;
	const uint8_t *rptr;

	wptr = reqbuff;
	put32bit(&wptr, CLTOMA_FUSE_GETTRASHTIME);
	put32bit(&wptr, kGetTrashTimePayloadSize);
	put32bit(&wptr, 0);
	putINode(&wptr, inode);
	put8bit(&wptr, mode);
	if (tcpwrite(fd, reqbuff, kReqBuffSize) != kReqBuffSize) {
		printf("%s: master query: send error\n", fname);
		close_master_conn(1);
		return -1;
	}
	if (tcpread(fd, reqbuff, 8) != 8) {
		printf("%s: master query: receive error\n", fname);
		close_master_conn(1);
		return -1;
	}
	rptr = reqbuff;
	get32bit(&rptr, cmd);
	get32bit(&rptr, leng);
	if (cmd != MATOCL_FUSE_GETTRASHTIME) {
		printf("%s: master query: wrong answer (type)\n", fname);
		close_master_conn(1);
		return -1;
	}
	buff = (uint8_t *)malloc(leng);
	if (tcpread(fd, buff, leng) != (int32_t)leng) {
		printf("%s: master query: receive error\n", fname);
		free(buff);
		close_master_conn(1);
		return -1;
	}
	close_master_conn(0);
	rptr = buff;
	get32bit(&rptr, cmd);  // queryid
	if (cmd != 0) {
		printf("%s: master query: wrong answer (queryid)\n", fname);
		free(buff);
		return -1;
	}
	leng -= 4;
	if (leng == 1) {
		printf("%s: %s\n", fname, saunafs_error_string(*rptr));
		free(buff);
		return -1;
	} else if (leng < 8 || leng % 8 != 0) {
		printf("%s: master query: wrong answer (leng)\n", fname);
		free(buff);
		return -1;
	} else if (mode == GMODE_NORMAL && leng != 16) {
		printf("%s: master query: wrong answer (leng)\n", fname);
		free(buff);
		return -1;
	}
	if (mode == GMODE_NORMAL) {
		get32bit(&rptr, fn);
		get32bit(&rptr, dn);
		get32bit(&rptr, trashtime);
		get32bit(&rptr, cnt);
		if ((fn != 0 || dn != 1) && (fn != 1 || dn != 0)) {
			printf("%s: master query: wrong answer (fn,dn)\n", fname);
			free(buff);
			return -1;
		}
		if (cnt != 1) {
			printf("%s: master query: wrong answer (cnt)\n", fname);
			free(buff);
			return -1;
		}
		printf("%s: %" PRIu32 "\n", fname, trashtime);
	} else {
		std::vector<std::pair<uint32_t, uint32_t>> files;
		std::vector<std::pair<uint32_t, uint32_t>> dirs;
		get32bit(&rptr, fn);
		get32bit(&rptr, dn);
		files.reserve(fn);
		dirs.reserve(dn);
		for (i = 0; i < fn; ++i) {
			get32bit(&rptr, trashtime);
			get32bit(&rptr, cnt);
			files.push_back({trashtime, cnt});
		}
		for (i = 0; i < dn; ++i) {
			get32bit(&rptr, trashtime);
			get32bit(&rptr, cnt);
			dirs.push_back({trashtime, cnt});
		}
		std::sort(files.begin(), files.end());
		std::sort(dirs.begin(), dirs.end());
		printf("%s:\n", fname);
		for (const auto &entry : files) {
			printf(" files with trashtime        %10" PRIu32 " :", entry.first);
			print_number(" ", "\n", entry.second, 1, 0, 1);
		}
		for (const auto &entry : dirs) {
			printf(" directories with trashtime  %10" PRIu32 " :", entry.first);
			print_number(" ", "\n", entry.second, 1, 0, 1);
		}
	}
	free(buff);
	return 0;
}

static int gene_get_trashtime_run(int argc, char **argv, int rflag) {
	int ch, status;
	while ((ch = getopt(argc, argv, "rnhH")) != -1) {
		switch (ch) {
		case 'n':
			humode = 0;
			break;
		case 'h':
			humode = 1;
			break;
		case 'H':
			humode = 2;
			break;
		case 'r':
			rflag = 1;
			break;
		}
	}
	argc -= optind;
	argv += optind;

	if (argc < 1) {
		get_trashtime_usage();
		return 1;
	}

	status = 0;
	while (argc > 0) {
		if (get_trashtime(*argv, (rflag) ? GMODE_RECURSIVE : GMODE_NORMAL) < 0) {
			status = 1;
		}
		argc--;
		argv++;
	}
	return status;
}

int get_trashtime_run(int argc, char **argv) {
	return gene_get_trashtime_run(argc, argv, 0);
}
