/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import { i18n, setupI18n } from "../utils/i18n";
import { ComponentChildren, createContext, h, VNode } from "preact";
import { useContext } from "preact/hooks";
import { Locale } from "date-fns";
import {
  es as esLocale,
  enGB as enLocale,
  fr as frLocale,
  de as deLocale,
} from "date-fns/locale";

export type InternationalizationAPI = typeof i18n;

interface Type {
  lang: string;
  supportedLang: { [id in keyof typeof supportedLang]: string };
  i18n: InternationalizationAPI;
  dateLocale: Locale;
  completeness: { [id in keyof typeof supportedLang]: number };
}

const supportedLang = {
  es: "Espanol [es]",
  en: "English [en]",
  fr: "Francais [fr]",
  de: "Deutsch [de]",
  sv: "Svenska [sv]",
  it: "Italiane [it]",
};

const initial: Type = {
  lang: "en",
  supportedLang,
  i18n,
  dateLocale: enLocale,
  completeness: {
    de: 0,
    en: 0,
    es: 0,
    fr: 0,
    it: 0,
    sv: 0,
  },
};
const Context = createContext<Type>(initial);

interface Props {
  children: ComponentChildren;
  lang: string;
  source: Record<string, any>;
  completeness?: Record<string, number>;
}

// Outmost UI wrapper.
export const TranslationProvider = ({
  children,
  lang,
  source,
  completeness: completenessProp,
}: Props): VNode => {
  const completeness = {
    en: 100,
    de:
      !completenessProp || !completenessProp["de"] ? 0 : completenessProp["de"],
    es:
      !completenessProp || !completenessProp["es"] ? 0 : completenessProp["es"],
    fr:
      !completenessProp || !completenessProp["fr"] ? 0 : completenessProp["fr"],
    it:
      !completenessProp || !completenessProp["it"] ? 0 : completenessProp["it"],
    sv:
      !completenessProp || !completenessProp["sv"] ? 0 : completenessProp["sv"],
  };

  setupI18n(lang, source);

  const dateLocale =
    lang === "es"
      ? esLocale
      : lang === "fr"
        ? frLocale
        : lang === "de"
          ? deLocale
          : enLocale;

  return h(Context.Provider, {
    value: {
      lang,
      supportedLang,
      i18n,
      dateLocale,
      completeness,
    },
    children,
  });
};

export const useTranslationContext = (): Type => useContext(Context);

const MIN_LANG_COVERAGE_THRESHOLD = 90;
/**
 * choose the best from the browser config based on the completeness
 * on the translation
 */
function getBrowserLang(
  completeness: Record<string, number>,
): string | undefined {
  if (typeof window === "undefined") return undefined;

  if (window.navigator.language) {
    if (
      completeness[window.navigator.language] >= MIN_LANG_COVERAGE_THRESHOLD
    ) {
      return window.navigator.language;
    }
  }
  if (window.navigator.languages) {
    const match = Object.entries(completeness)
      .filter(([code, value]) => {
        if (value < MIN_LANG_COVERAGE_THRESHOLD) return false; //do not consider langs below 90%
        return (
          window.navigator.languages.findIndex((l) => l.startsWith(code)) !== -1
        );
      })
      .map(([code, value]) => ({ code, value }));

    if (match.length > 0) {
      let max = match[0];
      match.forEach((v) => {
        if (v.value > max.value) {
          max = v;
        }
      });
      return max.code;
    }
  }

  return undefined;
}
