/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Duration,
  ScopeType,
  TalerCorebankApiClient,
  TalerMerchantInstanceHttpClient,
  j2s,
  succeedOrThrow,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { defaultCoinConfig } from "../harness/denomStructures.js";
import {
  createWalletDaemonWithClient,
  makeTestPaymentV2,
  withdrawViaBankV3,
} from "../harness/environments.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  HarnessExchangeBankAccount,
  MerchantService,
  getTestHarnessPaytoForLabel,
  setupDb,
} from "../harness/harness.js";

export async function runCurrencyScopeSeparationTest(t: GlobalTestState) {
  // Set up test environment
  const dbDefault = await setupDb(t);

  const dbExchangeTwo = await setupDb(t, {
    nameSuffix: "exchange2",
  });

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: dbDefault.connStr,
    httpPort: 8082,
  });

  const exchangeOne = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: dbDefault.connStr,
  });

  const exchangeTwo = ExchangeService.create(t, {
    name: "testexchange-2",
    currency: "TESTKUDOS",
    httpPort: 8281,
    database: dbExchangeTwo.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: dbDefault.connStr,
  });

  let exchangeOneBankAccount: HarnessExchangeBankAccount = {
    wireGatewayApiBaseUrl: new URL(
      "accounts/myexchange/taler-wire-gateway/",
      bank.corebankApiBaseUrl,
    ).href,
    wireGatewayAuth: {
      username: "myexchange",
      password: "password",
    },
    accountPaytoUri: getTestHarnessPaytoForLabel("myexchange"),
  };

  let exchangeTwoBankAccount: HarnessExchangeBankAccount = {
    wireGatewayApiBaseUrl: new URL(
      "accounts/myexchange2/taler-wire-gateway/",
      bank.corebankApiBaseUrl,
    ).href,
    wireGatewayAuth: {
      username: "myexchange2",
      password: "password",
    },
    accountPaytoUri: getTestHarnessPaytoForLabel("myexchange2"),
  };

  bank.setSuggestedExchange(
    exchangeOne,
    exchangeOneBankAccount.accountPaytoUri,
  );

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "admin-password",
    },
  });

  await bankClient.registerAccountExtended({
    name: exchangeOneBankAccount.wireGatewayAuth.username,
    username: exchangeOneBankAccount.wireGatewayAuth.username,
    password: exchangeOneBankAccount.wireGatewayAuth.password,
    is_taler_exchange: true,
    payto_uri: exchangeOneBankAccount.accountPaytoUri,
  });
  await exchangeOne.addBankAccount("1", exchangeOneBankAccount);

  await bankClient.registerAccountExtended({
    name: exchangeTwoBankAccount.wireGatewayAuth.username,
    username: exchangeTwoBankAccount.wireGatewayAuth.username,
    password: exchangeTwoBankAccount.wireGatewayAuth.password,
    is_taler_exchange: true,
    payto_uri: exchangeTwoBankAccount.accountPaytoUri,
  });
  await exchangeTwo.addBankAccount("1", exchangeTwoBankAccount);

  // Set up the first exchange

  exchangeOne.addOfferedCoins(defaultCoinConfig);
  await exchangeOne.start();
  await exchangeOne.pingUntilAvailable();

  // Set up the second exchange

  exchangeTwo.addOfferedCoins(defaultCoinConfig);
  await exchangeTwo.start();
  await exchangeTwo.pingUntilAvailable();

  // Start and configure merchant

  merchant.addExchange(exchangeOne);

  await merchant.start();
  await merchant.pingUntilAvailable();

  const { accessToken: adminAccessToken } =
    await merchant.addInstanceWithWireAccount({
      id: "admin",
      name: "Default Instance",
      paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
      defaultWireTransferDelay: Duration.toTalerProtocolDuration(
        Duration.fromSpec({ minutes: 1 }),
      ),
    });

  const { walletClient } = await createWalletDaemonWithClient(t, {
    name: "wallet",
  });

  console.log("setup done!");

  // Withdraw digital cash into the wallet.

  const w1 = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange: exchangeOne,
    amount: "TESTKUDOS:15",
  });

  const w2 = await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange: exchangeTwo,
    amount: "TESTKUDOS:1",
  });

  await w1.withdrawalFinishedCond;
  await w2.withdrawalFinishedCond;

  const bal = await walletClient.call(WalletApiOperation.GetBalances, {});
  console.log(j2s(bal));

  const payRes = await makeTestPaymentV2(t, {
    merchant,
    merchantAdminAccessToken: adminAccessToken,
    walletClient,
    order: {
      amount: "TESTKUDOS:10",
      summary: "Test",
    },
  });

  const merchantClient = new TalerMerchantInstanceHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  const refundResp = succeedOrThrow(
    await merchantClient.addRefund(adminAccessToken, payRes.orderId, {
      reason: "test",
      refund: "TESTKUDOS:5",
    }),
  );

  await walletClient.call(WalletApiOperation.StartRefundQueryForUri, {
    talerRefundUri: refundResp.taler_refund_uri,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const txns1 = await walletClient.call(WalletApiOperation.GetTransactionsV2, {
    scopeInfo: {
      type: ScopeType.Exchange,
      currency: "TESTKUDOS",
      url: exchangeOne.baseUrl,
    },
  });

  console.log(`txns1: ${j2s(txns1)}`);

  const txns2 = await walletClient.call(WalletApiOperation.GetTransactionsV2, {
    scopeInfo: {
      type: ScopeType.Exchange,
      currency: "TESTKUDOS",
      url: exchangeTwo.baseUrl,
    },
  });

  console.log(`txns2: ${j2s(txns2)}`);

  t.assertDeepEqual(
    txns1.transactions.map((x) => x.type),
    ["withdrawal", "payment", "refund"],
  );
  t.assertDeepEqual(
    txns2.transactions.map((x) => x.type),
    ["withdrawal"],
  );
}

runCurrencyScopeSeparationTest.suites = ["wallet"];
